/* This is a "Hello World" type of program for PAMR usage.
 * It creates 1D grid and evolves it for NSTEPS steps.
 * Each evolution step simply shifts the grid function by one
 * grid point to the right.
 */
#include <stdio.h>
#include <math.h>
#include <pamr.h>

#define DIM  1 // dimensionality of the problem
#define XMIN 0.0 
#define XMAX 1.0
#define NX   129  // total number of grid points in x-direction
#define NTLEV 2   // number of time levels
#define NMAXGRIDS 1 // maximum number of grids (1 for unigrid)
#define NMAXGF 1 // maximum number of grid functions
#define OUTMODULO 10 // save every OUTMODULO steps
#define NSTEPS 51 // number of evolution steps
#define NGHOSTS 1 // number of ghost zones

#define GF_F 0 // symbolic name of index for grid function "f"

// declare the fortran functions - note the trailing underscore
void initialize_(int *Nx,real *x,real *f,real *xmin,real *xmax);
void make_step_(int *Nx,real *x,real *f1,real *f2);

int main( int argc, char *argv[])
{
  int shape[DIM],cp_rank,pamr_context;
  real bbox[2*DIM],q,q_size;
  char *cp_file;
  int lev[NMAXGRIDS];
  real gbox[2*DIM*NMAXGRIDS];
  char *varname;
  int in_amrh,in_mgh,num_tl,amr_inject,amr_interp;
  int amr_bdy_interp,amr_sync,mg_inject,mg_interp;
  int mg_sync,mg_noinj_to_amr,regrid_transfer,phys_bdy_type[2*DIM];

  int pamr_sgfn[NMAXGF];

  int g_rank,g_dim,g_shape[DIM],g_ghost_width[2*DIM],g_ngfs;
  real g_bbox[2*DIM],g_t,*g_x[DIM],*g_gfs[NMAXGF*NTLEV];

  int valid,Ng[DIM],min_lev,max_lev,num,Nx;

  int step,nsteps,outmodulo,tlev1,tlev2,i;
  real dt,time;

  real *x,*f[NTLEV];


  bbox[0] = XMIN;
  bbox[1] = XMAX;
  shape[0] = NX;
  q = 0;
  q_size = 0;
  cp_file = (char *) NULL;
  cp_rank = -1;

  MPI_Init(&argc,&argv);
  pamr_context = PAMR_init_context(0,q_size,DIM,shape,bbox,cp_file,cp_rank);
  if (!pamr_context) {
    perror("Error initializing pamr_context");
    exit(-1);
  }
  // define variable attributes for unigrid (no transfers necessary)
  varname = (char *) strdup("f");
  in_amrh = 1;
  in_mgh = 0;
  num_tl = NTLEV;
  amr_inject = PAMR_NO_INJECT;
  amr_interp = PAMR_NO_INTERP;
  amr_bdy_interp = PAMR_NO_INTERP;
  amr_sync = PAMR_SYNC;
  mg_inject = PAMR_NO_INJECT;
  mg_interp = PAMR_NO_INTERP;
  mg_sync = PAMR_NO_SYNC;
  mg_noinj_to_amr = 0;
  regrid_transfer = PAMR_NO_TRANSFER;

  // note that sgfn starts at 1 therefore we subtract -1 for C-style arrays
  pamr_sgfn[GF_F] = PAMR_def_var_full(varname,in_amrh,in_mgh,num_tl,amr_inject,
              amr_interp,amr_bdy_interp,amr_sync,mg_inject,mg_interp,
              mg_sync,mg_noinj_to_amr,regrid_transfer,phys_bdy_type)-1;
  Ng[0] = NGHOSTS;
  PAMR_set_ghost_width(Ng);
  // for unigrid we have only level = 1 grid
  min_lev = 1;
  max_lev = 1;
  num = 1;
  lev[0] = 1; 
  gbox[0] = bbox[0];
  gbox[1] = bbox[1];
  time = 0.0;
  // compose the hierarchy
  PAMR_compose_hierarchy(min_lev,max_lev,num,lev,gbox,time);
  // for unigrid we do not have to loop over grids since there is only one grid
  valid = PAMR_init_s_iter(1,PAMR_AMRH,0);
  if (valid) {
    PAMR_get_g_attribs(&g_rank,&g_dim,g_shape,g_bbox,g_ghost_width,&g_t,&g_ngfs,g_x,g_gfs);
  }
  else {
    perror("Error PAMR_init_s_iter\n");
    exit(-1);
  }

  tlev1 = 0;
  tlev2 = 1;

  Nx = g_shape[0];
  x = g_x[0];
  f[tlev1] = g_gfs[pamr_sgfn[GF_F]];
  f[tlev2] = g_gfs[pamr_sgfn[GF_F]+1];
  initialize_(&Nx,x,f[tlev1],&bbox[0],&bbox[1]);

  dt = 1.0;
  step = 0;
  while (step<NSTEPS) {
    if (g_rank ==0) printf("step=%d\n",step);
    if ((step % OUTMODULO) == 0) {
      if (g_rank ==0) printf("SAVING at time=%g\n",time);
      if (!PAMR_save_gfn("f",PAMR_AMRH,1,1,time,"","")) {
        perror("Error saving hierarchy");
      }
    }
    make_step_(&Nx,x,f[tlev1],f[tlev2]);
    PAMR_sync(1,tlev2+1,PAMR_AMRH,0);
    step++;
    time += dt;
    // cycle time levels
    tlev1 = (tlev1 + 1) % NTLEV;    
    tlev2 = (tlev2 + 1) % NTLEV;    
  }
  PAMR_free_context(pamr_context);
}  
